/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::  Module      :   Module Definition File
::  Copyright   :   (C)2002-2009 Woodward
::  Platform(s) :   MPC5xx
::  Dependants  :   None
::
::-----------------------------------------------------------------------------------------------------------
::
::  API
::  Provides definitions of the various modules supported by the MPC5xx platform
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file  ModuleDefn.h
 \brief The Module Definition describes factory preset data preset at time of manufacture.*/

#ifndef __MODULEDEFN_H
#define __MODULEDEFN_H

/*---- INCLUDE FILES --------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <ModuleDefn_MPC5xx.h>
#include <ModuleDefn_MPC55xx.h>

/*---- DEFINES --------------------------------------------------------------------------------------------*/
/*===== THESE MUST NOT CHANGE AS THEY WILL AFFECT OLDER VERSIONS OF MODULECONFIG ==============*/
/*! \brief The maximum model number text length. */
#define MAX_MODEL_NUMBER            50
/*! \brief The maximum serial number text length. */
#define MAX_SERIAL_NUMBER           50
/*! \brief The maximum part number text length. */
#define MAX_PART_NUMBER             40


/*==== DOCUMENT MAJOR VERSIONS ONLY HERE ======================================================*/
#define PCM_MODULE                  0x08
#define ECM_MODULE                  0x09
#define PCM03_MODULE                0x0A
#define VCM03_MODULE                0x0B
#define PCMHD_MODULE                0x0C
#define PCMRACE_MODULE              0x0D
#define ISDL_555_MODULE             0x0E
#define ISDL_565_MODULE             0x0F

#define ECM48_MODULE_HWMAJ          0x13
#define AIM_MODULE_HWMAJ            0x14
#define FIRE48_MODULE_HWMAJ         0x15
#define TVM48_MODULE_HWMAJ          0x16

#define AXIOM_MPC5554_HWMAJ         0x17

#define INDY05_MPC5554_HWMAJ        0x18
#define PCM09xx_MPC5554_HWMAJ       0x19

/*==== END MAJOR VERSION DEFINITIONS ==========================================================*/

/*---- TYPEDEF --------------------------------------------------------------------------------------------*/
#pragma pack(1)

typedef struct
{
    uint4   BootCheck           :  1;   /* TRUE if the CRC record should be checked by the bootstrap code */
    uint4   ApplicationCheck    :  1;   /* TRUE if the CRC record should be checked by the application code */
    uint4   UNUSED              : 30;
} S_CRCFlags;

typedef struct
{
    S_CRCFlags  Flags;                  /* Flags controlling how the CRC record be interpreted */
    uint4*      pStartAddress;          /* Start address of the CRC block (MUST be aligned on 4 byte address boundary) */
    uint4*      pEndAddress;            /* End address of the CRC block (MUST be aligned on 4 byte address boundary) */
    uint4       CRCResult;              /* Result that the computation should be checked against */
} S_CRCEntry;

typedef struct
{
    uint4 NumEntries;                   /* Number of CRC records in the list */
    S_CRCEntry const* pEntryArr;        /* Pointer to the first entry in the array */
} S_CRCBlock;


/* Name: E_ModuleCapability ClassID:ENUMDEC EnumDflt:"Capability Unknown" */
typedef enum 
{
    MODULE_DEVELOPMENT = 0,            /* EnumTxt:"Calibratable Development Module" */
    MODULE_PRODUCTION = 1,             /* EnumTxt:"Non-Calibratable Production Module" */
    MODULE_SOFTWARE = 2,               /* EnumTxt:"Software Dev Module (with RAM overlay only)" */
    MODULE_SOFTWARE_TRACE = 3,         /* EnumTxt:"Software Dev Module (with Trace Pod & RAM)" */
    MODULE_DEVELOPMENT_UNCENSORED = 4, /* EnumTxt:"Calibratable Development Module with Censorship disabled" */
    MODULE_PRODUCTION_UNCENSORED = 5,  /* EnumTxt:"Non-Calibratable Production Module with Censorship disabled" */
    NUM_MODULE_CAPABILITIES
} E_ModuleCapability;

/*! \brief The CAN Bus parameters are used to define the characteristics of the bus and hence its baud rate.

    They are applied to the module's CAN hardware when the at startup. */
typedef struct
{
/*! \brief Prescaler divider factor.

    Defines the ratio between the CPI clock frequency and the serial clock (SCK) frequency. */
    uint1 u1PresDiv;
 
/*! \brief Propagation segment. 

    Defines the length of the propagation segment in the bit time. */
    uint1 u1PropSeg;

/*! \brief Phase segment 1. 

    Defines the length of phase buffer segment 1 in the bit time */
    uint1 u1PhaseSeg1;
/*! \brief Phase segment 2. 

    Defines the length of phase buffer segment 2 in the bit time */
    uint1 u1PhaseSeg2;

/*! \brief Resync jump width. 

    Defines the maximum number of time quanta1 that a bit time can be changed by one re-synchronization. */
    uint1 u1RJW;
} S_ModuleConfigCANBusParams;
#define S_ModuleConfigCANBusParams_CONSTRUCTOR(u1PresDiv,u1PropSeg,u1PhaseSeg1,u1PhaseSeg2,u1RJW) \
            { u1PresDiv,u1PropSeg,u1PhaseSeg1,u1PhaseSeg2,u1RJW }

/*! \brief The version 0 module configuration data.  This information is factory preset at the time
 of manufacture */
typedef struct 
{
    uint1 BootVerMaj;                       /*!< Bootstrap application major version */
    uint1 BootVerMin;                       /*!< Bootstrap application minor version */
    uint1 EcuHwMaj;                         /*!< ECU Hardware major version */
    uint1 EcuHwMin;                         /*!< ECU Hardware mainor version */
    uint1 Capability;                       /*!< ECU Hardware capability */
    char ModelNumber[MAX_MODEL_NUMBER];     /*!< NUL terminated string of the model number */
    char SerialNumber[MAX_SERIAL_NUMBER];   /*!< NUL terminated string of the serial number */
    uint1 DefaultCityID;                    /*!< The default City identifier for CAN communications */
} S_ModuleConfigV0;

/*! \brief The version 1 module configuration data.  This information is factory preset at the time
 of manufacture */
typedef struct 
{
    uint1 BootVerMaj;                       /*!< BootStrap major version */
    uint1 BootVerMin;                       /*!< BootStrap minor version */
    uint1 EcuHwMaj;                         /*!< ECU Hardware major version */
    uint1 EcuHwMin;                         /*!< ECU Hardware mainor version */
    uint1 Capability;                       /*!< ECU Hardware capability */
    char ModelNumber[MAX_MODEL_NUMBER];     /*!< NUL terminated string of the model number */
    char SerialNumber[MAX_SERIAL_NUMBER];   /*!< NUL terminated string of the serial number */
    uint1 DefaultCityID;                    /*!< The default City identifier for CAN communications */
    uint4 DefaultCANBitRate;                /*!< The default bit rate to use for for CAN communications */
    uint4 DefaultSerialBaudRate;            /*!< The default serial baud rate to use for for Serial communications */
    uint4 BootSystemClockSpeed;             /*!< The system clock speed to use for bootstrap applications */
} S_ModuleConfigV1;

/*! \brief The version 2 module configuration data.  This information is factory preset at the time
 of manufacture */
typedef struct 
{
    uint1 ConfigVerMaj;                     /*!< Module Configuration Application major version */
    uint1 ConfigVerMin;                     /*!< Module Configuration Application minor version */
    uint1 EcuHwMaj;                         /*!< ECU Hardware major version */
    uint1 EcuHwMin;                         /*!< ECU Hardware mainor version */
    uint1 Capability;                       /*!< ECU Hardware capability */
    char ModelNumber[MAX_MODEL_NUMBER];     /*!< NUL terminated string of the model number */
    char SerialNumber[MAX_SERIAL_NUMBER];   /*!< NUL terminated string of the serial number */
    uint1 DefaultCityID;                    /*!< The default City identifier for CAN communications */
    uint4 DefaultCANBitRate;                /*!< The default bit rate to use for for CAN communications */
    uint4 DefaultSerialBaudRate;            /*!< The default serial baud rate to use for for Serial communications */
    uint4 BootSystemClockSpeed;             /*!< The system clock speed to use for bootstrap applications */
    char PartNumber[MAX_PART_NUMBER];       /*!< NUL terminated string of the part number */
} S_ModuleConfigV2;

/*! \brief The version 3 module configuration data. This information is factory preset at the time
 of manufacture.*/
typedef struct 
{
    uint1 ConfigVerMaj;                   /*!< Module Configuration Application major version */
    uint1 ConfigVerMin;                   /*!< Module Configuration Application minor version */
    uint1 EcuHwMaj;                       /*!< ECU Hardware major version */
    uint1 EcuHwMin;                       /*!< ECU Hardware mainor version */
    uint1 Capability;                     /*!< ECU Hardware capability */
/*! The default CAN bus parameters */
    S_ModuleConfigCANBusParams CANBusParameters;
/*! Shift right the MessageID by this many bits and cast to a byte to extract SS */
    uint1 uRXIdShiftRightForSS;
/*! Shift right the MessageID by this many bits and cast to a byte to extract DD */
    uint1 uRXIdShiftRightForDD;
/*! Defines which boot loader {MotoTron or a Vendor should be used} */
    uint1 uBootLoaderID;
    char ModelNumber[50];                /*!< NUL terminated string of the model number */
    char SerialNumber[49];               /*!< NUL terminated string of the serial number */
    uint4 u32DefaultTXMessageID;         /*!< The Base message ID for TX */
    uint4 u32DefaultRXMessageID;         /*!< The Base message ID for RX. The acceptance mask may open this up */
    uint4 u32DefaultRXAcceptMask;        /*!< Apply this acceptance mask for received messages */
    uint4 u32DefaultTXAcceptMask;        /*!< Apply this acceptance mask for Transmitted messages */
    uint4 DefaultSerialBaudRate;         /*!< The default serial baud rate to use for for Serial communications */
    uint4 BootSystemClockSpeed;          /*!< The system clock speed to use for bootstrap applications */
    char PartNumber[40];                 /*!< NUL terminated string of the part number */
} S_ModuleConfigV3;
#define S_ModuleConfigV3_CONSTRUCTOR(ConfigVerMaj,                                              \
                                     ConfigVerMin,                                              \
                                     EcuHwMaj,                                                  \
                                     EcuHwMin,                                                  \
                                     Capability,                                                \
                                     u1PresDiv,                                                 \
                                     u1PropSeg,                                                 \
                                     u1PhaseSeg1,                                               \
                                     u1PhaseSeg2,                                               \
                                     u1RJW,                                                     \
                                     uRXIdShiftRightForSS,                                      \
                                     uRXIdShiftRightForDD,                                      \
                                     uBootLoaderID,                                             \
                                     ModelNumber,                                               \
                                     SerialNumber,                                              \
                                     u32DefaultTXMessageID,                                     \
                                     u32DefaultRXMessageID,                                     \
                                     u32DefaultRXAcceptMask,                                    \
                                     u32DefaultTXAcceptMask,                                    \
                                     DefaultSerialBaudRate,                                     \
                                     BootSystemClockSpeed,                                      \
                                     PartNumber)                                                \
             {  ConfigVerMaj,                                                                   \
                ConfigVerMin,                                                                   \
                EcuHwMaj,                                                                       \
                EcuHwMin,                                                                       \
                Capability,                                                                     \
                S_ModuleConfigCANBusParams_CONSTRUCTOR(u1PresDiv,u1PropSeg,u1PhaseSeg1,u1PhaseSeg2,u1RJW),  \
                uRXIdShiftRightForSS,                                                           \
                uRXIdShiftRightForDD,                                                           \
                uBootLoaderID,                                                                  \
                ModelNumber,                                                                    \
                SerialNumber,                                                                   \
                u32DefaultTXMessageID,                                                          \
                u32DefaultRXMessageID,                                                          \
                u32DefaultRXAcceptMask,                                                         \
                u32DefaultTXAcceptMask,                                                         \
                DefaultSerialBaudRate,                                                          \
                BootSystemClockSpeed,                                                           \
                PartNumber }

/*! \brief The module configuration.  The version number indicates which module configuration
 the pointer points to. */
typedef struct 
{
    uint4 u4Version;                        /*!< Module Configuration data struct version */
    void const* pModuleConfig;              /*!< Pointer to a module configuration struct of the above version */
    uint4 u4ModuleConfigSize;               /*!< Holds the size in bytes of the configuration pointed to by \ref pModuleConfig */
} S_ModuleConfigDefn;

#pragma pack()

/*---- FUNCTION PROTOTYPES --------------------------------------------------------------------------------*/

#endif /* __MODULEDEFN_H */

/*---- END OF FILE ----------------------------------------------------------------------------------------*/

